/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0, 0.0,0.0)

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// maybe something between 0.5 and 2.0
const float CLOUD_HEIGHT = 1.1;

// scale of clouds
const float UV_FREQ = 0.005;

// cloudiness, bigger number = less clouds
const float CLOUD_FILTER = 0.45;

// parallax layers
const int PARALLAX_LAYERS = 8;


float filter(float f, float a)
{
   f = clamp(f - a, 0.0, 1.0);
   f /= (1.0 - a);    
   return f;
}

float fbm(vec2 uv)
{
    float f = (texture2D(iChannel0, uv * 2.0).r - 0.5) * 0.2;
    f += (texture2D(iChannel0, uv * 4.0).r - 0.5) * 0.125;
    f += (texture2D(iChannel0, uv * 8.0).r - 0.5) * 0.125 * 0.5;
    f += (texture2D(iChannel0, uv * 16.0).r - 0.5) * 0.125 * 0.25;
    f += (texture2D(iChannel0, uv * 32.0).r - 0.5) * 0.125 * 0.24;
    f += (texture2D(iChannel0, uv * 64.0).r - 0.5) * 0.125 * 0.22;
    f += (texture2D(iChannel0, uv * 128.0).r - 0.5) * 0.125 * 0.12;
    f += (texture2D(iChannel0, uv * 256.0).r - 0.5) * 0.125 * 0.1;
    f += 0.5;
    return clamp(f, 0.0, 1.0);
}



vec2 getuv(in vec2 uv, float l)
{
    vec3 rd = normalize(vec3(uv, 0.4));
    vec2 _uv = vec2(rd.x / abs(rd.y) * l, rd.z / abs(rd.y) * l);
    return _uv;
}

// cloud rendering
void clouds (vec2 uv, inout vec4 col, float t, float freq)
{
    vec2 _uv = getuv(uv, 1.0);
    _uv.y += t;
    float l = 1.0;
    
    for (int i = 0; i < PARALLAX_LAYERS; ++i)
    {
        // 3 parallax layers of clouds
        float h = fbm(_uv * freq) * 0.5;
        h += fbm(vec2(-t * 0.001, t * 0.0015) + _uv * freq * 1.1) * 0.35;
        h += fbm(vec2(t * 0.001, -t * 0.0025) + _uv * freq * 1.2) * 0.15;
        
        float f = filter(h, CLOUD_FILTER);
        
        f -= (l - 1.0) * CLOUD_HEIGHT; // height
        f = clamp(f, 0.0, 1.0);
        
        col += f * vec4(0.9, 0.9, 1.0, 1.0) * (1.0 - col.a);
        
        l *= 1.1 - h * 0.18; // parallax control, offset uv by height
        _uv = getuv(uv, l);
    	_uv.y += t;
    }
}

// fbm for reflections
float wfbm(vec2 uv)
{
    float f = (texture2D(iChannel1, uv * 1.0).r - 0.5) * 0.5;
    f += (texture2D(iChannel1, uv * 2.0).r - 0.5) * 0.5 * 0.5;
    f += (texture2D(iChannel1, uv * 4.0).r - 0.5) * 0.25 * 0.5;
    f += (texture2D(iChannel1, uv * 8.0).r - 0.5) * 0.25 * 0.5 * 0.5;
    f += (texture2D(iChannel1, uv * 16.0).r - 0.5) * 0.25 * 0.5 * 0.5 * 0.5;
    return f + 0.5;
    
}

vec3 grad(vec2 uv)
{
	vec2 off = vec2(0.15, 0.0);
    vec3 g = vec3(wfbm(uv + off.xy) - wfbm(uv - off.xy),
                  off.x,
				  wfbm(uv + off.yx) - wfbm(uv - off.yx));
    
    return normalize(g);
}



void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    fragColor = vec4(0.0);
    
	vec2 uv = fragCoord.xy / iResolution.xy;
    uv -= vec2(0.5);
    uv.y /= iResolution.x / iResolution.y;
    
    vec4 dark = vec4(0.1, 0.2, 0.3, 0.0) * 1.5;
    vec4 light = vec4(0.3, 0.4, .55, 0.0) * 1.5;
    vec4 bg = mix(light, dark, abs(uv.y) * 6.5);
    vec4 col = vec4(0);
    
    vec2 _uv = uv;
    _uv.y -= iGlobalTime * 0.01;
    _uv.x *= 0.1;
    vec2 guv = vec2(0.0);
    
    if (uv.y < 0.0)
    {
	    vec3 g = grad(_uv * 5.0);
	    guv = vec2(g.x / g.y, g.z / g.y);
    }
    
    clouds(uv + guv * 0.015 * mix(-0.0, 1.0, clamp(abs(uv.y) * 5.0 - 0.04, 0.0, 1.0)  ), col, iGlobalTime * 0.4, UV_FREQ);
   
    fragColor = mix(bg, col, col.a);

    // some graphical candy, sun halos etc.
    if(uv.y < 0.0)
    {
        _uv = uv;
    	fragColor = mix(dark, fragColor, (1.0 - smoothstep(-0.05, -0., uv.y) * 0.75));
        
        _uv.x *= 0.1;
        fragColor = mix(fragColor, vec4(1.0), 1.0 - smoothstep(0.0, 0.025, length(_uv)));

        _uv.y *= 0.05;
        _uv.x *= 0.35;
        fragColor = mix(fragColor, vec4(1.0), 1.0 - smoothstep(0.0, 0.005, length(_uv)));
        
    }else
    {
	    vec4 wcolor = light * 1.3;
    	fragColor = mix(wcolor, fragColor, (smoothstep(0., .1, uv.y)));        
		vec2 _uv = uv;	
        _uv.x *= 0.1;
        fragColor = mix(fragColor, vec4(1.0), 1.0 - smoothstep(0.0, 0.1, length(_uv)));
    }
    
    uv.x *= 0.015;
    fragColor = mix(fragColor, vec4(1.0), 1.0 - smoothstep(0., 0.01, length(uv)));
    fragColor = mix(fragColor, vec4(1.0), 1.0 - smoothstep(0., 0.005, length(uv)));
    
    
}


void main ( void )
{
  mainImage( gl_FragColor, gl_FragCoord.xy );
}
